using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Drawing;
using System.Text;

namespace TitleExtractor
{
    static class Program
    {
        [STAThread]
        static void Main()
        {
            ApplicationConfiguration.Initialize();
            Application.Run(new MainForm());
        }
    }

    public class MainForm : Form
    {
        TextBox inputBox;
        TextBox outputBox;
        Button extractBtn;
        Button copyBtn;
        Button saveBtn;
        Button clearBtn;
        CheckBox keepDuplicatesCheck;
        CheckBox showStepsCheck;
        Label statusLabel;

        public MainForm()
        {
            Text = "Nicjo’s Title Extractor — Credits to Clean Titles";
            MinimumSize = new Size(800, 500);
            ClientSize = new Size(980, 720);
            StartPosition = FormStartPosition.CenterScreen;
            Font = new Font("Segoe UI", 9);
            BackColor = Color.FromArgb(248, 248, 248);

            // --- Input Label ---
            var lbl1 = new Label()
            {
                Text = "Paste raw tracklist here (left):",
                Left = 12,
                Top = 10,
                Width = 420
            };
            Controls.Add(lbl1);

            // --- Input Box ---
            inputBox = new TextBox()
            {
                Multiline = true,
                Left = 12,
                Top = 30,
                Width = 460,
                Height = 570,
                ScrollBars = ScrollBars.Vertical,
                Font = new Font("Segoe UI", 10),
                Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left
            };
            Controls.Add(inputBox);

            // Enable drag & drop for input
            inputBox.AllowDrop = true;
            inputBox.DragEnter += (s, e) =>
            {
                if (e.Data.GetDataPresent(DataFormats.FileDrop))
                    e.Effect = DragDropEffects.Copy;
            };
            inputBox.DragDrop += (s, e) =>
            {
                try
                {
                    string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);
                    if (files.Length > 0)
                    {
                        inputBox.Text = System.IO.File.ReadAllText(files[0], Encoding.UTF8);
                        statusLabel.Text = $"Loaded: {System.IO.Path.GetFileName(files[0])}";
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Could not load file: " + ex.Message);
                }
            };

            // --- Output Label ---
            var lbl2 = new Label()
            {
                Text = "Extracted titles (one per line):",
                Left = 490,
                Top = 10,
                Width = 420
            };
            Controls.Add(lbl2);

            // --- Output Box ---
            outputBox = new TextBox()
            {
                Multiline = true,
                Left = 490,
                Top = 30,
                Width = 470,
                Height = 570,
                ScrollBars = ScrollBars.Vertical,
                Font = new Font("Segoe UI", 10),
                ReadOnly = true,
                Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right
            };
            Controls.Add(outputBox);

            // --- Tip label ---
            var tip = new Label()
            {
                Text = "Tip: Paste or drag any messy tracklist (Discogs, Japanese lines, times, credits). Click Extract.",
                Left = 12,
                Top = ClientSize.Height - 115,
                Width = 930,
                ForeColor = Color.DimGray,
                Font = new Font("Segoe UI", 8.75f, FontStyle.Italic),
                Anchor = AnchorStyles.Bottom | AnchorStyles.Left
            };
            Controls.Add(tip);

            // --- Footer panel ---
            var footer = new Panel()
            {
                Height = 90,
                Dock = DockStyle.Bottom,
                BackColor = Color.FromArgb(240, 240, 240)
            };
            Controls.Add(footer);

            // --- Flat buttons ---
            extractBtn = new Button() { Text = "Extract Titles →", Width = 150, Height = 34, Left = 10, Top = 18 };
            copyBtn = new Button() { Text = "Copy Output", Width = 120, Height = 34, Left = 170, Top = 18 };
            saveBtn = new Button() { Text = "Save .txt...", Width = 120, Height = 34, Left = 300, Top = 18 };
            clearBtn = new Button() { Text = "Clear", Width = 80, Height = 34, Left = 430, Top = 18 };

            foreach (var btn in new[] { extractBtn, copyBtn, saveBtn, clearBtn })
            {
                btn.FlatStyle = FlatStyle.Flat;
                btn.FlatAppearance.BorderSize = 0;
                btn.BackColor = Color.WhiteSmoke;
                btn.Cursor = Cursors.Hand;
                btn.MouseEnter += (s, e) => ((Button)s).BackColor = Color.Gainsboro;
                btn.MouseLeave += (s, e) => ((Button)s).BackColor = Color.WhiteSmoke;
            }

            footer.Controls.AddRange(new Control[] { extractBtn, copyBtn, saveBtn, clearBtn });

            // --- Checkboxes (right side) ---
            keepDuplicatesCheck = new CheckBox()
            {
                Text = "Keep Dupes",
                AutoSize = true,
                Left = 540,
                Top = 25
            };
            showStepsCheck = new CheckBox()
            {
                Text = "Show debug steps",
                AutoSize = true,
                Left = 650,
                Top = 25
            };
            footer.Controls.AddRange(new Control[] { keepDuplicatesCheck, showStepsCheck });

            // --- Status label ---
            statusLabel = new Label()
            {
                Text = "Ready.",
                AutoSize = true,
                ForeColor = Color.Gray,
                Left = 12,
                Top = 52
            };
            footer.Controls.Add(statusLabel);

            // --- Event hookups ---
            extractBtn.Click += ExtractBtn_Click;
            copyBtn.Click += (s, e) =>
            {
                if (!string.IsNullOrEmpty(outputBox.Text))
                {
                    Clipboard.SetText(outputBox.Text);
                    statusLabel.Text = "Copied to clipboard.";
                }
            };
            saveBtn.Click += (s, e) =>
            {
                using (var sfd = new SaveFileDialog() { Filter = "Text files|*.txt", FileName = "titles.txt" })
                {
                    if (sfd.ShowDialog() == DialogResult.OK)
                    {
                        System.IO.File.WriteAllText(sfd.FileName, outputBox.Text, new UTF8Encoding(true));
                        statusLabel.Text = $"Saved: {System.IO.Path.GetFileName(sfd.FileName)}";
                    }
                }
            };
            clearBtn.Click += (s, e) =>
            {
                inputBox.Clear();
                outputBox.Clear();
                statusLabel.Text = "Cleared.";
            };
        }

        private void ExtractBtn_Click(object sender, EventArgs e)
        {
            try
            {
                var raw = inputBox.Text;
                var result = ExtractTitles(raw, showStepsCheck.Checked, out List<string> debugLines);

                if (!keepDuplicatesCheck.Checked)
                    result = result.Distinct(StringComparer.OrdinalIgnoreCase).ToList();

                result = result.Where(t => !string.IsNullOrWhiteSpace(t)).ToList();

                outputBox.Text = string.Join(Environment.NewLine, result);
                statusLabel.Text = $"Extracted {result.Count} titles.";

                if (showStepsCheck.Checked)
                {
                    var dlg = new Form() { Width = 820, Height = 560, Text = "Debug / Extraction Steps" };
                    var tb = new TextBox() { Multiline = true, Dock = DockStyle.Fill, ScrollBars = ScrollBars.Vertical, ReadOnly = true, Font = new Font("Consolas", 9) };
                    tb.Text = string.Join(Environment.NewLine, debugLines);
                    dlg.Controls.Add(tb);
                    dlg.Show();
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show("Error: " + ex.Message);
            }
        }

        static List<string> ExtractTitles(string input, bool debug, out List<string> debugLines)
        {
            debugLines = new List<string>();
            var titles = new List<string>();
            if (string.IsNullOrWhiteSpace(input)) return titles;

            input = input.Replace("—", "-").Replace("–", "-").Replace("\t", "    ");
            var rawLines = Regex.Split(input, @"\r\n|\r|\n").Select(l => l.Trim()).Where(l => l.Length > 0).ToList();
            if (debug) debugLines.Add($"Total non-empty lines: {rawLines.Count}");

            var timePattern = new Regex(@"\b\d{1,2}:\d{2}\b");
            var leadingTrackNumber = new Regex(@"^\s*\d{1,2}(?:[-.]\d{1,2})?(?:\s*[).:-])?\s*");
            var creditsPattern = new Regex(@"\b(?:accompanied|arranged|producer|written|engineer|vocals|conductor|percussion|drums|guitar|piano|bass|strings|label|format|released|country|genre|style|voice|recorded)\b", RegexOptions.IgnoreCase);
            var parenthesesOnly = new Regex(@"^\(.+\)$");

            foreach (var lineRaw in rawLines)
            {
                string line = lineRaw;
                if (creditsPattern.IsMatch(line)) continue;
                if (timePattern.IsMatch(line) && line.Length <= 6) continue;
                line = Regex.Replace(line, @"\s+\d{1,2}:\d{2}\s*$", "");

                if (line.Contains("-"))
                {
                    var parts = line.Split(new[] { " - ", "-" }, StringSplitOptions.None)
                        .Select(p => p.Trim()).Where(p => p.Length > 0).ToList();
                    if (parts.Count >= 2)
                    {
                        var candidate = parts.Last();
                        if (candidate.Length < 2 && parts.Count >= 2) candidate = parts[^2];
                        if (!timePattern.IsMatch(candidate))
                        {
                            titles.Add(candidate);
                            continue;
                        }
                    }
                }

                var noNumber = leadingTrackNumber.Replace(line, "").Trim();
                if (!string.Equals(noNumber, line, StringComparison.Ordinal))
                {
                    var candidate = Regex.Replace(noNumber, @"\s+\d{1,2}:\d{2}\s*$", "").Trim();
                    if (candidate.Length > 0 && !creditsPattern.IsMatch(candidate))
                    {
                        titles.Add(candidate);
                        continue;
                    }
                }

                var columns = Regex.Split(line, @"\s{2,}").Select(c => c.Trim()).Where(c => c.Length > 0).ToList();
                if (columns.Count >= 2)
                {
                    var candidate = Regex.Replace(columns.Last(), @"\s+\d{1,2}:\d{2}\s*$", "").Trim();
                    if (!creditsPattern.IsMatch(candidate) && !parenthesesOnly.IsMatch(candidate))
                    {
                        titles.Add(candidate);
                        continue;
                    }
                }

                if (line.Length > 0 && line.Length < 120 && !creditsPattern.IsMatch(line))
                {
                    var candidate = leadingTrackNumber.Replace(line, "").Trim();
                    candidate = Regex.Replace(candidate, @"^[\-\:\.\)]+", "").Trim();
                    candidate = Regex.Replace(candidate, @"\s+\d{1,2}:\d{2}\s*$", "").Trim();
                    if (candidate.Length > 0 && !creditsPattern.IsMatch(candidate))
                    {
                        titles.Add(candidate);
                        continue;
                    }
                }
            }

            return titles.Select(t => Regex.Replace(t.Trim().Trim('\"', '\'', '‘', '’'), @"\s{2,}", " "))
                         .Where(t => t.Length > 0).ToList();
        }
    }
}
